/*
 * Die Sourcecodes, die diesem Buch als Beispiele beiliegen, sind
 * Copyright (c) 2006 - Thomas Ekert. Alle Rechte vorbehalten.
 * 
 * Trotz sorgfltiger Kontrolle sind Fehler in Softwareprodukten nie vollstndig auszuschlieen.
 * Die Sourcodes werden in Ihrem Originalzustand ausgeliefert.
 * Ansprche auf Anpassung, Weiterentwicklung, Fehlerbehebung, Support
 * oder sonstige wie auch immer gearteten Leistungen oder Haftung sind ausgeschlossen.
 * Sie drfen kommerziell genutzt, weiterverarbeitet oder weitervertrieben werden.
 * Voraussetzung hierfr ist, dass fr jeden beteiligten Entwickler, jeweils mindestens
 * ein Exemplar dieses Buches in seiner aktuellen Version als gekauftes Exemplar vorliegt.
 */
package djbuch.kapitel_11; import java.io.*; import java.util.Locale;

import jxl.*; import jxl.write.*; import jxl.read.biff.BiffException;
import lotus.domino.*;
/**
 * Einfache Implementierung einer Klasse, die es ermglicht, Excel Sheets in
 * RichText Tabellen zu konvertieren und umgekehrt.
 * Hierbei werden insbesondere Techniken aufgezeigt, die notwendig sind,
 * um innerhalb von Tabellen in den Zellen zu navigieren.
 * Die verschiedenen Methoden zeigen darberhinaus allgemeine Techniken,
 * die bei der Verarbeitung von Tabellen zur Anwendung kommen
 * Die Excel Operationen basieren auf dem Framework von Andy Khan
 * http://www.andykhan.com/jexcelapi/index.html
 * 
 * @author Thomas Ekert
 */
public class ExcelToRichTextConverter {

	private String fileName;
	private RichTextItem richText;
	private String encoding=null;
	private Session session = null;
	private ColorObject color = null;
	private ColorObject altColor = null;	
	
	/**
	 * Objekt, anhand dessen das bergebene RichTextItem fr den Import oder Export
	 * in oder aus diesem Item genutzt werden kann.
	 * @param rti
	 * @param useFileName
	 * @throws NotesException
	 */
	public ExcelToRichTextConverter(RichTextItem rti, String useFileName) throws NotesException {
		fileName = useFileName;
		richText = rti;
		session = rti.getParent().getParentDatabase().getParent();
		color = session.createColorObject();
		color.setNotesColor(RichTextStyle.COLOR_GRAY);
		altColor = session.createColorObject();
		altColor.setNotesColor(RichTextStyle.COLOR_LIGHT_GRAY);
	}
	
	/** Fgt eine Tabelle ans Ende des RichTextItems und fllt diese mit dem
	 * Inhalt der Exceltabelle.
	 * @throws BiffException
	 * @throws IOException
	 * @throws NotesException
	 */
	public void runImport() throws BiffException, IOException, NotesException {
		Workbook book = Workbook.getWorkbook(new File(fileName));
		Sheet xlsSheet = book.getSheet(0);

		richText.appendTable(xlsSheet.getRows(), xlsSheet.getColumns());
		RichTextTable rtt = getTable(getTableCount());  //letzte Tabelle im rti ist die neue Tabelle
		setColors (rtt,true);
		RichTextNavigator rtnav = richText.createNavigator();
		int allCellCount = getCellCount(rtnav);
		if (allCellCount <= 0) {
			return;
		}
		rtnav.findNthElement(RichTextItem.RTELEM_TYPE_TABLECELL, allCellCount
				- (rtt.getRowCount() * rtt.getColumnCount()) + 1);
		for (int i = 0; i < rtt.getRowCount(); i++) {
			Cell[] row = xlsSheet.getRow(i);
			for (int j = 0; j < rtt.getColumnCount(); j++) {
				richText.beginInsert(rtnav);
				if (j<row.length) {
					richText.appendText(row[j].getContents());
				} else {
					richText.appendText("");
				}
				richText.endInsert();
				rtnav.findNextElement();
			}
		}
		book.close();
	}
	
	/**
	 * exportiert das RichTextItem (field richText) in eine Datei (field fileName).
	 * @param noOfTableToExport - laufende Nummer der Tabelle, die exportiert werden soll, beginnend bei 1
	 * @return - true, falls alles ok
	 */
    public boolean runExport(int noOfTableToExport) {
		try {
			RichTextTable rtt = getTable(noOfTableToExport);
			int colCount = rtt.getColumnCount();
			int rowCount = rtt.getRowCount();
			WorkbookSettings wbSet = new WorkbookSettings();
		    wbSet.setLocale(new Locale("de", "DE"));
			WritableWorkbook book = Workbook.createWorkbook(new File(fileName), wbSet);
			WritableSheet xlsSheet = book.createSheet("NotesRichText", 0);
			RichTextRange tableAsRange = richText.createRange();
			tableAsRange.setBegin(rtt);
			tableAsRange.setEnd(rtt);
			RichTextNavigator tableCellNavigator = tableAsRange.getNavigator();
			RichTextRange cellContent = richText.createRange();
			tableCellNavigator.findFirstElement(RichTextItem.RTELEM_TYPE_TABLECELL);
			for (int rows = 0; rows<rowCount; rows++) {
				for (int cols = 0; cols<colCount; cols++) {
					cellContent.setBegin(tableCellNavigator);
					cellContent.setEnd(tableCellNavigator);
					String content = cellContent.getTextRun();
					xlsSheet.addCell(new Label (cols,rows,content));
					tableCellNavigator.findNextElement();
				}
			}
			xlsSheet.addCell(new Label(0, rowCount + 2,
					"Dieses Excel Sheet wurde aus dem Domino Dokument \""
							+ richText.getParent()
									.getItemValueString("F_titel")
							+ "\" mit Universal ID "
							+ richText.getParent().getUniversalID()
							+ " erzeugt."));
			book.write();
			book.close();
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
		return true;
	}
    
	
	/**
	 * Liefert die Gesamtzahl aller Tabellenzellen aller tabellen im RichText
	 * @param nav
	 * @return
	 * @throws NotesException
	 */
	private int getCellCount (RichTextNavigator nav) throws NotesException {
		int result = -1;
		if (!nav.findFirstElement(RichTextItem.RTELEM_TYPE_TABLECELL)) {
			return 0;
		} else {
			result = 1;
		}
		while (nav.findNextElement()) {result++;}
		return result;
	}
	
	/**
	 * Setzt in der Tabelle rtt die in der Klasse verankerten Farben color und altColor
	 * Zustzlich wird die tabelle ber deren Eigenschaften so eingestellt, dass
	 * entwder die Spalten oder die Zeilen abwechselnde Farben erhalten. 
	 * @param rtt
	 * @param alternateColumns - Abwechselnde Spaltenfarben, falls true, sonst abwechselnde Zeilenfarben
	 * @throws NotesException
	 */
	private void setColors (RichTextTable rtt, boolean alternateColumns) throws NotesException{
		rtt.setColor (color);
		rtt.setAlternateColor(altColor);
		if (alternateColumns) {
			rtt.setStyle (RichTextTable.TABLESTYLE_ALTERNATINGCOLS);
		} else {
			rtt.setStyle (RichTextTable.TABLESTYLE_ALTERNATINGROWS);
		}
	}
	
	/**
	 * liefert die Tabelle an Position noOfTable. Falls diese grer als die 
	 * Anzahl der Tabellen im RTItem, dann wird die letzte Tabelle geliefert.
	 * @param noOfTable
	 * @return
	 * @throws NotesException
	 */
	public RichTextTable getTable(int noOfTable) throws NotesException {
		RichTextNavigator rtn = rtn = richText.createNavigator();
		boolean ok = rtn.findFirstElement(RichTextItem.RTELEM_TYPE_TABLE);
		if (!ok)
			return null;
		for (int i = 1; i < noOfTable; i++) {
			rtn.findNextElement();
		}
		return (RichTextTable) rtn.getElement();
	}
	
	/**
	 * liefert die Anzahl der Tabellen im RichTextItem (field richText)
	 * 
	 * @return
	 * @throws NotesException
	 */
	public int getTableCount () throws NotesException {
		RichTextNavigator rtn = richText.createNavigator();
		boolean ok = rtn.findFirstElement(RichTextItem.RTELEM_TYPE_TABLE);
		if (!ok) return 0;
		int result = 1;
		while (ok) {
			ok = rtn.findNextElement();
			if (ok) result++;
		}
		return result;
	}
    
	public void setEncoding (String enc) {
		encoding = enc;
	}
	
	public String getEncoding () {
		return encoding;
	}
	
	public String getFileName() {
		return fileName;
	}
	
	public void setFileName(String name) {
		fileName = name;
	}
}